#!/usr/bin/python
# -*- coding: iso-8859-2 -*-

###########################################################################
# IA-32 assembly compiler for a Simple Programming Language
#
# Copyright (C) 2006-2008 Gerg RDI
# http://cactus.rulez.org/
#
# Published under the terms of the GNU General Public License 2.0
# $Id: sp.py 188 2006-03-12 11:59:46Z cactus $
###########################################################################
#
# SP is a toy language allowing the following simple statements:
#
# READ var              Read an int from the user, and store it in 'var'
# WRITE expr            Print the value of 'expr'
# GOTO lbl [IF expr]    Jump to the specified line. If 'expr' is present,
#                       jump only if the result of 'expr' >= 0
# LET var = expr        Set value of 'var' to 'expr'
#
# There are 33 variables, named X, Y, Z, X0, ..., X9, Y0, ...
# The type of variables and expressions are all 32-bit signed integers
# 
###########################################################################

import sp
import parser

class Compiler:
    def __init__ (self):
        self.variables = {}

    def writeline (self, s = ""):
        print s
    
    def var_address (self, varname):
        if not varname in self.variables:
            self.variables[varname] = len (self.variables)
        return '_sp_v + %d' % (self.variables[varname] * 4)

    def label (self, lbl):
        return '_sp_l_%s' % lbl

    def get_evaluator (self):
        class CompileTimeEvaluator:
            def get_variable (self, var):
                raise sp.CannotEvaluateYetError ()
        return CompileTimeEvaluator ()

    def compile (self, program):
        def create_var (varname):
            self.writeline ('%s\tresd 1' % self.var_address (varname))

        # String literals
        self.writeline ('\tsection .data')
        self.writeline ('_sp_fmt\t\tdb "%d", 0x0A, 0x00')
        self.writeline ('_sp_prompt\tdb "> ", 0x00')
        self.writeline ('_sp_infmt\tdb "%d", 0x00')
        self.writeline ()

        # Asm code of the program itself ('main' is exported, so you can link it with GCC)
        self.writeline ('\tsection .text')
        self.writeline ('\tglobal main')
        self.writeline ('\textern printf, scanf')
        self.writeline ('main:')
        
        for (lbl, stmt) in program:
            if not (lbl is None):
                self.writeline ('%s:' % self.label (lbl))
            for asmline in stmt.compile_to_asm (self):
                self.writeline ('\t%s' % asmline)
            self.writeline ()
        self.writeline ('\tret')

        # Implementation of _sp_input
        self.writeline ("""
_sp_input:
	push ebp
	mov ebp, esp
	push dword 0

_sp_input_loop:
	push _sp_prompt
	call printf
	sub esp, 4

	push ebp
	push _sp_infmt
	call scanf
	sub esp, 8

	mov edx, [ebp]
	cmp eax, 1 		; TODO: real input checking
	jne _sp_input_loop

	mov esp, ebp
	pop ebp
	ret
	""")

        # Space for SP variables
        self.writeline ('\tsection .bss')
        self.writeline ('_sp_v\tresd %d' % len (self.variables))

def compile_sp (filename):
    prog = parser.load_sp (filename)
    compiler = Compiler ()
    compiler.compile (prog)

if __name__ == '__main__':
    import sys

    for filename in sys.argv[1:]:
        compile_sp (filename)
